<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

include(__DIR__ . '/includes/functions.php');
$static_url = dirname($_SERVER['SCRIPT_NAME']) . '/assets/';
include 'session_check.php';

// Obter domínio base dinamicamente
$protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http";
$domain = $_SERVER['HTTP_HOST'];
$base_url = $protocol . '://' . $domain;

// Configurações do arquivo JSON do aplicativo 📁
$configFile = __DIR__ . '/api/app_info.json';
$configData = [
    'latest_version' => 1,
    'apk_url' => '',
    'force_update' => false
];

// Carrega configurações existentes 📤
if (file_exists($configFile)) {
    $configContent = file_get_contents($configFile);
    $loadedData = json_decode($configContent, true);
    if (is_array($loadedData)) {
        $configData = array_merge($configData, $loadedData);
    }
}

// Processamento do formulário 📝
$success_message = "";
$error_message = "";

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    try {
        $latest_version = isset($_POST['latest_version']) ? intval($_POST['latest_version']) : 1;
        $force_update = isset($_POST['force_update']) ? true : false;
        
        // Processar upload do APK
        $apk_url = $configData['apk_url']; // Mantém a URL atual por padrão
        
        if (isset($_FILES['apk_file']) && $_FILES['apk_file']['error'] === UPLOAD_ERR_OK) {
            $apk_file = $_FILES['apk_file'];
            
            // Verificar se é um arquivo APK
            $file_info = pathinfo($apk_file['name']);
            if (strtolower($file_info['extension']) !== 'apk') {
                throw new Exception("⚠️ Apenas arquivos APK são permitidos");
            }
            
            // Criar diretório se não existir
            $apk_dir = __DIR__ . '/launcher/apk/';
            if (!is_dir($apk_dir)) {
                mkdir($apk_dir, 0755, true);
            }
            
            // Gerar nome único para o arquivo
            $new_filename = 'launcher_v' . $latest_version . '.apk';
            $destination = $apk_dir . $new_filename;
            
            // Remover versões anteriores se existirem
            $old_files = glob($apk_dir . 'launcher_v*.apk');
            foreach ($old_files as $old_file) {
                if (is_file($old_file) && basename($old_file) !== $new_filename) {
                    unlink($old_file);
                }
            }
            
            // Mover arquivo enviado
            if (move_uploaded_file($apk_file['tmp_name'], $destination)) {
                // Gerar URL completa com domínio
                $script_path = dirname($_SERVER['SCRIPT_NAME']);
                $apk_url = $base_url . $script_path . '/launcher/apk/' . $new_filename;
                
                // Verificar se a URL está correta
                if (!filter_var($apk_url, FILTER_VALIDATE_URL)) {
                    // Se falhar, usar caminho relativo
                    $apk_url = $script_path . '/launcher/apk/' . $new_filename;
                }
            } else {
                throw new Exception("❌ Erro ao fazer upload do arquivo APK. Verifique permissões da pasta.");
            }
        } else if (empty($configData['apk_url']) && empty($_POST['keep_current_apk'])) {
            throw new Exception("⚠️ É necessário fazer upload de um arquivo APK ou manter o atual");
        }
        
        // Validações
        if ($latest_version < 1) {
            throw new Exception("⚠️ A versão deve ser maior que 0");
        }
        
        // Opção para manter APK atual
        if (isset($_POST['keep_current_apk']) && !empty($configData['apk_url'])) {
            // Mantém a URL atual, apenas atualiza o nome do arquivo na URL
            $current_url = $configData['apk_url'];
            $new_url = preg_replace('/launcher_v\d+\.apk/', 'launcher_v' . $latest_version . '.apk', $current_url);
            $apk_url = $new_url;
        }

        // Atualizar configurações
        $configData['latest_version'] = $latest_version;
        $configData['apk_url'] = $apk_url;
        $configData['force_update'] = $force_update;

        // Salvar no arquivo JSON 💾
        if (file_put_contents($configFile, json_encode($configData, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES))) {
            $success_message = "✅ Configurações do aplicativo atualizadas com sucesso!";
            
            // Se houve upload de APK, adicionar mensagem extra
            if (isset($_FILES['apk_file']) && $_FILES['apk_file']['error'] === UPLOAD_ERR_OK) {
                $success_message .= "<br>📁 APK salvo em: " . htmlspecialchars($destination);
                $success_message .= "<br>🔗 URL gerada: " . htmlspecialchars($apk_url);
            }
        } else {
            throw new Exception("❌ Erro ao salvar configurações. Verifique as permissões do arquivo.");
        }
    } catch (Exception $e) {
        $error_message = $e->getMessage();
    }
}

$page_title = "📱 Gerenciamento do Aplicativo";

$page_content = '<div class="container mt-4 mt-md-5">
    <div class="card mb-4 mb-xl-8">
        <div class="card-body p-4 p-md-8 py-md-12">
            <!-- Cabeçalho Responsivo -->
            <div class="d-flex flex-column flex-md-row align-items-center mb-6 mb-md-7">
                <div class="symbol symbol-60px symbol-md-50px me-0 me-md-5 mb-4 mb-md-0">
                    <span class="symbol-label bg-light-primary">
                        <i class="ki-outline ki-mobile fs-2x fs-md-1hx text-primary"></i>
                    </span>
                </div>
                <div class="text-center text-md-start">
                    <h2 class="mb-1 fs-2hx fs-md-2">📱 Gerenciamento do Launcher APK</h2>
                    <p class="text-gray-600 fs-6 fs-md-5">Atualize a versão e arquivo do aplicativo</p>
                </div>
            </div>';

// Mensagens de Sucesso/Erro
if (!empty($success_message)) {
    $page_content .= '<div class="alert alert-success d-flex align-items-center mb-6" role="alert">
        <i class="ki-outline ki-check-circle fs-2x text-success me-4 d-none d-md-block"></i>
        <div class="d-flex flex-column">
            <h4 class="mb-1 text-success fs-5 fs-md-4">Sucesso!</h4>
            <span class="fs-6 fs-md-5">' . $success_message . '</span>
        </div>
    </div>';
}

if (!empty($error_message)) {
    $page_content .= '<div class="alert alert-danger d-flex align-items-center mb-6" role="alert">
        <i class="ki-outline ki-cross-circle fs-2x text-danger me-4 d-none d-md-block"></i>
        <div class="d-flex flex-column">
            <h4 class="mb-1 text-danger fs-5 fs-md-4">Erro!</h4>
            <span class="fs-6 fs-md-5">' . htmlspecialchars($error_message) . '</span>
        </div>
    </div>';
}

$page_content .= '<form method="POST" action="" enctype="multipart/form-data">
    <div class="row g-4 g-md-6 mb-8 mb-md-10">
        <div class="col-12">
            <div class="card card-flush">
                <div class="card-header py-4 py-md-6">
                    <h3 class="card-title fs-4 fs-md-3">⚙️ Configurações do Aplicativo</h3>
                </div>
                <div class="card-body pt-1 px-4 px-md-6 pb-4 pb-md-6">
                    
                    <!-- Versão do Aplicativo -->
                    <div class="mb-6 mb-md-8">
                        <label for="latest_version" class="fs-6 fs-md-5 fw-semibold mb-2 d-block">
                            <span class="d-inline-flex align-items-center">
                                <i class="ki-outline ki-number fs-3 me-2"></i>
                                <span>🔢 Versão do Aplicativo</span>
                            </span>
                        </label>
                        <input type="number" class="form-control form-control-lg form-control-solid" 
                            id="latest_version" name="latest_version" 
                            value="' . htmlspecialchars($configData['latest_version']) . '"
                            min="1" step="1" required
                            style="font-size: 1.1rem;">
                        <div class="form-text mt-2 fs-6 fs-md-5">Número da versão (apenas números inteiros)</div>
                    </div>
                    
                    <!-- Upload do APK -->
                    <div class="mb-6 mb-md-8">
                        <label for="apk_file" class="fs-6 fs-md-5 fw-semibold mb-2 d-block">
                            <span class="d-inline-flex align-items-center">
                                <i class="ki-outline ki-document-upload fs-3 me-2"></i>
                                <span>📁 Upload do APK</span>
                            </span>
                        </label>
                        <div class="input-group input-group-lg">
                            <input type="file" class="form-control form-control-solid" 
                                id="apk_file" name="apk_file" 
                                accept=".apk">
                        </div>
                        <div class="form-text mt-2 fs-6 fs-md-5">
                            Envie o arquivo APK do launcher. Será salvo como: launcher_v[versão].apk
                        </div>
                        
                        <!-- Opção para manter APK atual -->
                        ' . (!empty($configData['apk_url']) ? '
                        <div class="form-check form-check-custom form-check-solid mt-4">
                            <input class="form-check-input" type="checkbox" 
                                id="keep_current_apk" name="keep_current_apk" value="1">
                            <label class="form-check-label fs-6 fs-md-5" for="keep_current_apk">
                                <strong>Manter APK atual</strong> (apenas atualizar número da versão na URL)
                            </label>
                        </div>' : '') . '
                        
                        <!-- Mostrar APK atual se existir -->
                        ' . (!empty($configData['apk_url']) ? '
                        <div class="mt-5 p-4 bg-light-info rounded">
                            <h5 class="text-gray-800 mb-3 fs-5 fs-md-4">
                                <i class="ki-outline ki-document fs-3 me-2"></i>
                                📦 APK Atual:
                            </h5>
                            <div class="d-flex flex-column flex-md-row align-items-start align-items-md-center">
                                <div class="mb-3 mb-md-0 me-md-4">
                                    <a href="' . htmlspecialchars($configData['apk_url']) . '" 
                                       class="btn btn-primary btn-sm d-inline-flex align-items-center"
                                       target="_blank">
                                        <i class="ki-outline ki-download fs-4 me-2"></i>
                                        Baixar APK
                                    </a>
                                </div>
                                <div class="flex-grow-1">
                                    <p class="mb-1">
                                        <strong class="d-block d-md-inline">Arquivo:</strong> 
                                        <span class="text-primary ms-md-2">' . basename(parse_url($configData['apk_url'], PHP_URL_PATH)) . '</span>
                                    </p>
                                    <p class="mb-0 text-break">
                                        <strong class="d-block d-md-inline">URL:</strong> 
                                        <small class="ms-md-2">' . htmlspecialchars($configData['apk_url']) . '</small>
                                    </p>
                                </div>
                            </div>
                        </div>' : '') . '
                    </div>
                    
                    <!-- Forçar Atualização -->
                    <div class="mb-6 mb-md-8">
                        <div class="form-check form-switch form-check-custom form-check-solid d-flex align-items-center">
                            <input class="form-check-input" type="checkbox" 
                                id="force_update" name="force_update" 
                                style="width: 3.5rem; height: 1.75rem;"
                                ' . ($configData['force_update'] ? 'checked' : '') . '>
                            <label class="form-check-label ms-3" for="force_update">
                                <strong class="fs-6 fs-md-5">⚠️ Forçar Atualização</strong>
                            </label>
                        </div>
                        <div class="form-text mt-2 fs-6 fs-md-5">
                            Quando ativado, todos os usuários serão forçados a atualizar para esta versão
                        </div>
                    </div>
                    
                    <!-- Informações -->
                    <div class="notice d-flex flex-column flex-md-row bg-light-warning rounded border-warning border border-dashed p-4 p-md-6 mt-6">
                        <div class="d-none d-md-block me-4 me-md-6">
                            <i class="ki-outline ki-information-5 fs-2x text-warning"></i>
                        </div>
                        <div class="flex-grow-1">
                            <h4 class="text-gray-900 fw-bold mb-3 fs-5 fs-md-4">ℹ️ Como Funciona</h4>
                            <div class="fs-6 fs-md-5 text-gray-700">
                                <p class="mb-3"><strong>📱 Processo de Atualização:</strong></p>
                                <div class="row">
                                    <div class="col-12 col-md-6">
                                        <ul class="mb-3 mb-md-0">
                                            <li class="mb-2">1️⃣ Aumente o número da versão</li>
                                            <li class="mb-2">2️⃣ Faça upload do novo APK</li>
                                            <li class="mb-2">3️⃣ Arquivo salvo em: <code class="bg-light p-1 rounded">/launcher/apk/</code></li>
                                        </ul>
                                    </div>
                                    <div class="col-12 col-md-6">
                                        <ul>
                                            <li class="mb-2">4️⃣ Versões antigas são removidas</li>
                                            <li class="mb-2">5️⃣ <code>app_info.json</code> é atualizado</li>
                                            <li class="mb-2">6️⃣ URL completa é gerada automaticamente</li>
                                        </ul>
                                    </div>
                                </div>
                                <div class="mt-4 p-3 bg-light rounded">
                                    <p class="mb-2"><strong>🌐 Domínio Detectado:</strong></p>
                                    <code class="d-block p-2 bg-dark text-light rounded">' . htmlspecialchars($base_url) . '</code>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Botões Responsivos -->
    <div class="d-flex flex-column flex-md-row justify-content-between gap-3 mt-6 mt-md-8">
        <button type="submit" class="btn btn-primary btn-lg py-3 order-2 order-md-1">
            <span class="d-inline-flex align-items-center justify-content-center">
                <i class="ki-outline ki-check fs-2 me-2"></i>
                <span class="fs-6">💾 Salvar Configurações</span>
            </span>
        </button>
        
        <a href="javascript:history.back()" class="btn btn-secondary btn-lg py-3 order-1 order-md-2">
            <span class="d-inline-flex align-items-center justify-content-center">
                <i class="ki-outline ki-arrow-left fs-2 me-2"></i>
                <span class="fs-6">Voltar</span>
            </span>
        </a>
    </div>
</form>

<div class="separator separator-dashed my-8 my-md-10"></div>

<!-- Status Atual -->
<div class="row">
    <div class="col-12">
        <h4 class="mb-4 fs-4 fs-md-3">📋 Status Atual</h4>
        <div class="card card-flush">
            <div class="card-body p-4 p-md-6">
                <div class="row g-4">
                    <!-- Versão -->
                    <div class="col-12 col-md-4">
                        <div class="d-flex align-items-center">
                            <div class="symbol symbol-50px me-4 flex-shrink-0">
                                <span class="symbol-label bg-light-info">
                                    <i class="ki-outline ki-number fs-2x text-info"></i>
                                </span>
                            </div>
                            <div class="flex-grow-1">
                                <span class="text-gray-600 fw-semibold d-block fs-6 fs-md-5">Versão Atual</span>
                                <span class="text-gray-800 fw-bold d-block fs-3 fs-md-2hx mt-1">v' . htmlspecialchars($configData['latest_version']) . '</span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Atualização Forçada -->
                    <div class="col-12 col-md-4">
                        <div class="d-flex align-items-center">
                            <div class="symbol symbol-50px me-4 flex-shrink-0">
                                <span class="symbol-label bg-light-' . ($configData['force_update'] ? 'danger' : 'success') . '">
                                    <i class="ki-outline ki-' . ($configData['force_update'] ? 'warning-2' : 'check') . ' fs-2x text-' . ($configData['force_update'] ? 'danger' : 'success') . '"></i>
                                </span>
                            </div>
                            <div class="flex-grow-1">
                                <span class="text-gray-600 fw-semibold d-block fs-6 fs-md-5">Atualização Forçada</span>
                                <span class="text-gray-800 fw-bold d-block fs-3 fs-md-2hx mt-1">
                                    ' . ($configData['force_update'] ? 'ATIVADA ⚠️' : 'DESATIVADA ✅') . '
                                </span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- APK Disponível -->
                    <div class="col-12 col-md-4">
                        <div class="d-flex align-items-center">
                            <div class="symbol symbol-50px me-4 flex-shrink-0">
                                <span class="symbol-label bg-light-primary">
                                    <i class="ki-outline ki-download fs-2x text-primary"></i>
                                </span>
                            </div>
                            <div class="flex-grow-1">
                                <span class="text-gray-600 fw-semibold d-block fs-6 fs-md-5">APK Disponível</span>
                                <span class="text-gray-800 fw-bold d-block fs-3 fs-md-2hx mt-1">
                                    ' . (!empty($configData['apk_url']) ? 'SIM 📦' : 'NÃO ❌') . '
                                </span>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Prévia do JSON -->
                <div class="mt-8">
                    <h5 class="mb-3 fs-5 fs-md-4">
                        <i class="ki-outline ki-code-square fs-4 me-2"></i>
                        📄 Prévia do JSON Gerado:
                    </h5>
                    <div class="bg-dark text-light p-3 p-md-4 rounded">
                        <div class="overflow-auto" style="max-height: 300px;">
                            <pre class="mb-0" style="font-size: 0.875rem;"><code>' . json_encode($configData, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES) . '</code></pre>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener("DOMContentLoaded", function() {
    // Toggle entre upload e manter APK atual
    const keepCurrentCheckbox = document.getElementById("keep_current_apk");
    const fileInput = document.getElementById("apk_file");
    
    if (keepCurrentCheckbox && fileInput) {
        keepCurrentCheckbox.addEventListener("change", function() {
            fileInput.disabled = this.checked;
            fileInput.required = !this.checked;
        });
    }
    
    // Mostrar preview do nome do arquivo
    if (fileInput) {
        fileInput.addEventListener("change", function() {
            if (this.files.length > 0) {
                const versionInput = document.getElementById("latest_version");
                const filename = this.files[0].name;
                alert("Arquivo selecionado: " + filename + "\n\nSerá salvo como: launcher_v" + versionInput.value + ".apk");
            }
        });
    }
});
</script>

<style>
/* Estilos responsivos adicionais */
@media (max-width: 768px) {
    .symbol {
        width: 40px !important;
        height: 40px !important;
    }
    
    .symbol-label i {
        font-size: 1.5rem !important;
    }
    
    .btn-lg {
        padding: 0.75rem 1.5rem !important;
        font-size: 1rem !important;
    }
    
    .form-control-lg {
        padding: 0.75rem 1rem !important;
        font-size: 1rem !important;
    }
    
    .card {
        border-radius: 0.5rem !important;
    }
    
    .card-body {
        padding: 1rem !important;
    }
}

/* Melhorias para telas muito pequenas */
@media (max-width: 576px) {
    .container {
        padding-left: 0.75rem;
        padding-right: 0.75rem;
    }
    
    h2 {
        font-size: 1.5rem !important;
    }
    
    h3 {
        font-size: 1.25rem !important;
    }
    
    .alert {
        padding: 0.75rem !important;
    }
}

/* Melhorias de legibilidade */
.text-break {
    word-break: break-all;
    overflow-wrap: break-word;
}

/* Ajuste para inputs em mobile */
input[type="number"], 
input[type="file"] {
    font-size: 16px !important; /* Previne zoom em iOS */
}

/* Ajuste para o switch em mobile */
.form-check-input {
    min-width: 45px;
}

/* Ajuste para código em mobile */
pre code {
    font-size: 12px !important;
    line-height: 1.4;
}
</style>';

include 'includes/layout.php';
?>